/*
 * Decompiled with CFR 0.152.
 */
package device.airspy;

import common.Log;
import decoder.SourceUSB;
import device.DCRemovalFilter_RB;
import device.DeviceException;
import device.DevicePanel;
import device.HilbertTransform;
import device.ThreadPoolManager;
import device.TunerClass;
import device.TunerConfiguration;
import device.TunerController;
import device.airspy.AirspyDeviceInformation;
import device.airspy.AirspyPanel;
import device.airspy.AirspySampleAdapter;
import device.airspy.AirspySampleRate;
import device.airspy.AirspyTunerConfiguration;
import device.rtl.RTL2832TunerController;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.usb.UsbException;
import org.apache.commons.io.EndianUtils;
import org.usb4java.Device;
import org.usb4java.DeviceDescriptor;
import org.usb4java.DeviceHandle;
import org.usb4java.DeviceList;
import org.usb4java.LibUsb;
import org.usb4java.LibUsbException;
import org.usb4java.Transfer;
import org.usb4java.TransferCallback;

public class AirspyDevice
extends TunerController {
    public static final Gain LINEARITY_GAIN_DEFAULT = Gain.LINEARITY_14;
    public static final Gain SENSITIVITY_GAIN_DEFAULT = Gain.SENSITIVITY_10;
    public static final int GAIN_MIN = 1;
    public static final int GAIN_MAX = 22;
    public static final int LNA_GAIN_MIN = 0;
    public static final int LNA_GAIN_MAX = 14;
    public static final int LNA_GAIN_DEFAULT = 7;
    public static final int MIXER_GAIN_MIN = 0;
    public static final int MIXER_GAIN_MAX = 15;
    public static final int MIXER_GAIN_DEFAULT = 9;
    public static final int IF_GAIN_MIN = 0;
    public static final int IF_GAIN_MAX = 15;
    public static final int IF_GAIN_DEFAULT = 9;
    public static final long FREQUENCY_MIN = 24000L;
    public static final long FREQUENCY_MAX = 1800000L;
    public static final long FREQUENCY_DEFAULT = 101100000L;
    public static final double USABLE_BANDWIDTH_PERCENT = 0.9;
    public static final AirspySampleRate DEFAULT_SAMPLE_RATE = new AirspySampleRate(1, 3000000, "3.00 MHz");
    public static final long USB_TIMEOUT_MS = 2000L;
    public static final byte USB_ENDPOINT = -127;
    public static final byte USB_INTERFACE = 0;
    public static final int TRANSFER_BUFFER_POOL_SIZE = 16;
    public static final byte USB_REQUEST_IN = -64;
    public static final byte USB_REQUEST_OUT = 64;
    public static final DecimalFormat MHZ_FORMATTER = new DecimalFormat("#.00 MHz");
    private Device mDevice;
    private DeviceHandle mDeviceHandle;
    private ThreadPoolManager mThreadPoolManager;
    private LinkedTransferQueue<byte[]> mFilledBuffers = new LinkedTransferQueue();
    private int mBufferSize = 262144;
    private BufferProcessor mBufferProcessor = new BufferProcessor();
    private AirspySampleAdapter mSampleAdapter = new AirspySampleAdapter();
    private DCRemovalFilter_RB mDCFilter = new DCRemovalFilter_RB(0.01f);
    private HilbertTransform mHilbertTransform = new HilbertTransform();
    private AirspyDeviceInformation mDeviceInfo;
    private List<AirspySampleRate> mSampleRates = new ArrayList<AirspySampleRate>();
    private int mSampleRate = 0;
    SourceUSB usbSource;

    public AirspyDevice(Device device, ThreadPoolManager threadPoolManager) throws DeviceException {
        super("Airspy", 24000L, 1800000L);
        this.mDevice = device;
        this.name = "USBAirspy";
        this.mThreadPoolManager = threadPoolManager;
    }

    @Override
    public void setUsbSource(SourceUSB usb) {
        this.usbSource = usb;
        if (this.mBufferProcessor == null || !this.mBufferProcessor.isRunning()) {
            this.mBufferProcessor = new BufferProcessor();
            Thread thread = new Thread(this.mBufferProcessor);
            thread.setUncaughtExceptionHandler(Log.uncaughtExHandler);
            thread.setDaemon(true);
            thread.setName("Airspy Buffer Processor");
            thread.start();
        }
    }

    public void stop() {
        this.mBufferProcessor.stop();
    }

    public static AirspyDevice makeDevice() throws UsbException {
        DeviceList deviceList = new DeviceList();
        AirspyDevice dev = null;
        int result = LibUsb.init(null);
        if (result != 0) {
            Log.errorDialog("ERROR", "unable to initialize libusb [" + LibUsb.errorName(result) + "]");
        } else {
            Log.println("LibUSB API Version: " + LibUsb.getApiVersion());
            Log.println("LibUSB Version: " + LibUsb.getVersion());
            result = LibUsb.getDeviceList(null, deviceList);
            if (result < 0) {
                Log.errorDialog("ERROR", "unable to get device list from libusb [" + result + " / " + LibUsb.errorName(result) + "]");
            } else {
                Log.println("discovered [" + result + "] attached USB devices");
            }
        }
        for (Device device : deviceList) {
            DeviceDescriptor descriptor;
            result = LibUsb.getDeviceDescriptor(device, descriptor = new DeviceDescriptor());
            if (result != 0) {
                Log.errorDialog("ERROR", "unable to read device descriptor [" + LibUsb.errorName(result) + "]");
                continue;
            }
            if (device == null || descriptor == null) continue;
            TunerClass tunerClass = TunerClass.valueOf(descriptor.idVendor(), descriptor.idProduct());
            switch (tunerClass) {
                case AIRSPY: {
                    return AirspyDevice.initAirspyTuner(device, descriptor);
                }
            }
        }
        LibUsb.freeDeviceList(deviceList, true);
        return dev;
    }

    private static AirspyDevice initAirspyTuner(Device device, DeviceDescriptor descriptor) throws UsbException {
        try {
            ThreadPoolManager mThreadPoolManager = new ThreadPoolManager();
            AirspyDevice airspyController = new AirspyDevice(device, mThreadPoolManager);
            airspyController.init();
            return airspyController;
        }
        catch (DeviceException se) {
            Log.errorDialog("couldn't construct Airspy controller/tuner", se.getMessage());
            return null;
        }
    }

    public void init() throws DeviceException {
        this.mDeviceHandle = new DeviceHandle();
        int result = LibUsb.open(this.mDevice, this.mDeviceHandle);
        if (result != 0) {
            throw new DeviceException("Couldn't open airspy device - " + LibUsb.strError(result));
        }
        try {
            this.claimInterface();
        }
        catch (Exception e) {
            throw new DeviceException("Airspy Tuner Controller - error while setting USB configuration, claiming USB interface or reset the kernel mode driver\n" + e.getMessage());
        }
        try {
            this.setSamplePacking(false);
        }
        catch (LibUsbException e) {
            Log.println("Sample packing is not supported by airspy firmware");
        }
        catch (DeviceException e) {
            Log.println("Sample packing is not supported by airspy firmware");
        }
        catch (UnsupportedOperationException e) {
            Log.println("Sample packing is not supported by airspy firmware");
        }
        try {
            this.setReceiverMode(true);
        }
        catch (Exception e) {
            Log.errorDialog("Couldn't enable airspy receiver mode", e.getMessage());
        }
        this.setFrequency(101100000L);
        try {
            this.determineAvailableSampleRates();
        }
        catch (LibUsbException e) {
            Log.errorDialog("Error identifying available samples rates", e.getMessage());
        }
        catch (DeviceException e) {
            Log.errorDialog("Error identifying available samples rates", e.getMessage());
        }
        try {
            this.setSampleRate(DEFAULT_SAMPLE_RATE);
        }
        catch (IllegalArgumentException e) {
            Log.errorDialog("Setting sample rate is not supported by firmware", e.getMessage());
        }
        catch (LibUsbException e) {
            Log.errorDialog("Setting sample rate is not supported by firmware", e.getMessage());
        }
        catch (UsbException e) {
            Log.errorDialog("Setting sample rate is not supported by firmware", e.getMessage());
        }
    }

    private void claimInterface() throws DeviceException {
        if (this.mDeviceHandle != null) {
            int result = LibUsb.kernelDriverActive(this.mDeviceHandle, 0);
            if (result == 1 && (result = LibUsb.detachKernelDriver(this.mDeviceHandle, 0)) != 0) {
                Log.errorDialog("ERROR", "failed attempt to detach kernel driver [" + LibUsb.errorName(result) + "]");
                throw new DeviceException("couldn't detach kernel driver from device");
            }
            result = LibUsb.setConfiguration(this.mDeviceHandle, 1);
            if (result != 0) {
                throw new DeviceException("couldn't set USB configuration 1 [" + LibUsb.errorName(result) + "]");
            }
            result = LibUsb.claimInterface(this.mDeviceHandle, 0);
            if (result != 0) {
                throw new DeviceException("couldn't claim usb interface [" + LibUsb.errorName(result) + "]");
            }
        } else {
            throw new DeviceException("couldn't claim usb interface - no device handle");
        }
    }

    public static String getTransferStatus(int status) {
        switch (status) {
            case 0: {
                return "TRANSFER COMPLETED (0)";
            }
            case 1: {
                return "TRANSFER ERROR (1)";
            }
            case 2: {
                return "TRANSFER TIMED OUT (2)";
            }
            case 3: {
                return "TRANSFER CANCELLED (3)";
            }
            case 4: {
                return "TRANSFER STALL (4)";
            }
            case 5: {
                return "TRANSFER NO DEVICE (5)";
            }
            case 6: {
                return "TRANSFER OVERFLOW (6)";
            }
        }
        return "UNKNOWN TRANSFER STATUS (" + status + ")";
    }

    public void apply(TunerConfiguration config) throws DeviceException, LibUsbException, UsbException {
        if (config instanceof AirspyTunerConfiguration) {
            AirspyTunerConfiguration airspy = (AirspyTunerConfiguration)config;
            int sampleRate = airspy.getSampleRate();
            AirspySampleRate rate = this.getSampleRate(sampleRate);
            if (rate == null) {
                rate = DEFAULT_SAMPLE_RATE;
            }
            try {
                this.setSampleRate(rate);
            }
            catch (DeviceException e) {
                throw new DeviceException("Couldn't set sample rate [" + rate.toString() + "]" + e.getMessage());
            }
            try {
                this.setIFGain(airspy.getIFGain());
                this.setMixerGain(airspy.getMixerGain());
                this.setLNAGain(airspy.getLNAGain());
                this.setMixerAGC(airspy.isMixerAGC());
                this.setLNAAGC(airspy.isLNAAGC());
                this.setGain(airspy.getGain());
            }
            catch (Exception e) {
                throw new DeviceException("Couldn't apply gain settings from airspy config " + e.getMessage());
            }
            try {
                this.setFrequency(airspy.getFrequency());
            }
            catch (DeviceException deviceException) {}
        } else {
            throw new IllegalArgumentException("Invalid tuner config:" + config.getClass());
        }
    }

    @Override
    public int setFrequency(long frequency) throws DeviceException {
        if (24000L <= (frequency /= 1000L) && frequency <= 1800000L) {
            ByteBuffer buffer = ByteBuffer.allocateDirect(4);
            buffer.order(ByteOrder.LITTLE_ENDIAN);
            buffer.putInt((int)frequency);
            buffer.rewind();
            try {
                this.write(Command.SET_FREQUENCY, 0, 0, buffer);
            }
            catch (DeviceException e) {
                Log.errorDialog("error setting frequency [" + frequency + "]", e.getMessage());
                throw new DeviceException("error setting frequency [" + frequency + "]/n" + e.getMessage());
            }
        } else {
            throw new DeviceException("Frequency [" + frequency + "] outside " + "of tunable range " + 24000L + "-" + 1800000L);
        }
        return 0;
    }

    @Override
    public int getCurrentSampleRate() throws DeviceException {
        return this.mSampleRate;
    }

    public void setSampleRate(AirspySampleRate rate) throws LibUsbException, UsbException, DeviceException {
        if (rate.getRate() != this.mSampleRate) {
            int result = this.readByte(Command.SET_SAMPLE_RATE, 0, rate.getIndex(), true);
            if (result != 1) {
                throw new UsbException("Error setting sample rate [" + rate + "] rate - return value [" + result + "]");
            }
            this.mSampleRate = rate.getRate();
        }
    }

    public List<AirspySampleRate> getSampleRates() {
        return this.mSampleRates;
    }

    public AirspySampleRate getAirspySampleRate() {
        return this.getSampleRate(this.mSampleRate);
    }

    public AirspySampleRate getSampleRate(int rate) {
        for (AirspySampleRate sampleRate : this.mSampleRates) {
            if (sampleRate.getRate() != rate) continue;
            return sampleRate;
        }
        return null;
    }

    public void setSamplePacking(boolean enabled) throws LibUsbException, DeviceException {
        int result = this.readByte(Command.SET_PACKING, 0, enabled ? 1 : 0, true);
        if (result != 1) {
            throw new DeviceException("Couldnt set sample packing enabled: " + enabled);
        }
        this.mSampleAdapter.setSamplePacking(enabled);
    }

    public void setMixerAGC(boolean enabled) throws LibUsbException, DeviceException {
        int result = this.readByte(Command.SET_MIXER_AGC, 0, enabled ? 1 : 0, true);
        if (result != 0) {
            throw new DeviceException("Couldnt set mixer AGC enabled: " + enabled);
        }
    }

    public void setLNAAGC(boolean enabled) throws LibUsbException, DeviceException {
        int result = this.readByte(Command.SET_LNA_AGC, 0, enabled ? 1 : 0, true);
        if (result != 0) {
            throw new DeviceException("Couldnt set LNA AGC enabled: " + enabled);
        }
    }

    public void setGain(Gain gain) throws DeviceException {
        if (gain != Gain.CUSTOM) {
            this.setMixerAGC(false);
            this.setLNAAGC(false);
            this.setLNAGain(gain.getLNA());
            this.setMixerGain(gain.getMixer());
            this.setIFGain(gain.getIF());
        }
    }

    public void setLNAGain(int gain) throws LibUsbException, DeviceException, IllegalArgumentException {
        if (gain >= 0 && gain <= 14) {
            int result = this.readByte(Command.SET_LNA_GAIN, 0, gain, true);
            if (result != 0) {
                throw new DeviceException("Couldnt set LNA gain to: " + gain);
            }
        } else {
            throw new IllegalArgumentException("LNA gain value [" + gain + "] is outside value range: " + 0 + "-" + 14);
        }
    }

    public void setMixerGain(int gain) throws LibUsbException, DeviceException, IllegalArgumentException {
        if (gain >= 0 && gain <= 15) {
            int result = this.readByte(Command.SET_MIXER_GAIN, 0, gain, true);
            if (result != 0) {
                throw new DeviceException("Couldnt set mixer gain to: " + gain);
            }
        } else {
            throw new IllegalArgumentException("Mixer gain value [" + gain + "] is outside value range: " + 0 + "-" + 15);
        }
    }

    public void setIFGain(int gain) throws LibUsbException, DeviceException, IllegalArgumentException {
        if (gain >= 0 && gain <= 15) {
            int result = this.readByte(Command.SET_VGA_GAIN, 0, gain, true);
            if (result != 0) {
                throw new DeviceException("Couldnt set VGA gain to: " + gain);
            }
        } else {
            throw new IllegalArgumentException("VGA gain value [" + gain + "] is outside value range: " + 0 + "-" + 15);
        }
    }

    public void setReceiverMode(boolean enabled) throws LibUsbException, DeviceException {
        ByteBuffer buffer = ByteBuffer.allocateDirect(0);
        this.write(Command.RECEIVER_MODE, enabled ? 1 : 0, 0, buffer);
    }

    private void determineAvailableSampleRates() throws LibUsbException, DeviceException {
        this.mSampleRates.clear();
        this.mSampleRates.add(DEFAULT_SAMPLE_RATE);
        try {
            byte[] rawCount = this.readArray(Command.GET_SAMPLE_RATES, 0, 0, 4);
            if (rawCount != null) {
                int count = EndianUtils.readSwappedInteger(rawCount, 0);
                byte[] rawRates = this.readArray(Command.GET_SAMPLE_RATES, 0, count, count * 4);
                int x = 0;
                while (x < count) {
                    int rate = EndianUtils.readSwappedInteger(rawRates, x * 4);
                    if (rate != DEFAULT_SAMPLE_RATE.getRate()) {
                        this.mSampleRates.add(new AirspySampleRate(x, rate, AirspyDevice.formatSampleRate(rate)));
                    }
                    ++x;
                }
            }
        }
        catch (LibUsbException libUsbException) {
            // empty catch block
        }
    }

    private static String formatSampleRate(int rate) {
        return MHZ_FORMATTER.format((double)rate / 1000000.0);
    }

    public AirspyDeviceInformation getDeviceInfo() {
        if (this.mDeviceInfo == null) {
            this.readDeviceInfo();
        }
        return this.mDeviceInfo;
    }

    private void readDeviceInfo() {
        if (this.mDeviceInfo == null) {
            this.mDeviceInfo = new AirspyDeviceInformation();
        }
        try {
            int boardID = this.readByte(Command.BOARD_ID_READ, 0, 0, true);
            this.mDeviceInfo.setBoardID(boardID);
        }
        catch (DeviceException | LibUsbException e) {
            Log.errorDialog("Error reading airspy board ID", e.getMessage());
        }
        try {
            byte[] version = this.readArray(Command.VERSION_STRING_READ, 0, 0, 127);
            this.mDeviceInfo.setVersion(version);
        }
        catch (DeviceException | LibUsbException e) {
            Log.errorDialog("Error reading airspy version string", e.getMessage());
        }
        try {
            byte[] serial = this.readArray(Command.BOARD_PART_ID_SERIAL_NUMBER_READ, 0, 0, 24);
            this.mDeviceInfo.setPartAndSerialNumber(serial);
        }
        catch (DeviceException | LibUsbException e) {
            Log.errorDialog("Error reading airspy version string", e.getMessage());
        }
    }

    private int readByte(Command command, int value, int index, boolean signed) throws LibUsbException, DeviceException {
        if (this.mDeviceHandle != null) {
            ByteBuffer buffer = ByteBuffer.allocateDirect(1);
            int transferred = LibUsb.controlTransfer(this.mDeviceHandle, (byte)-64, command.getValue(), (short)value, (short)index, buffer, 2000L);
            if (transferred < 0) {
                throw new LibUsbException("read error", transferred);
            }
            byte result = buffer.get(0);
            if (signed) {
                return result & 0xFF;
            }
            return result;
        }
        throw new LibUsbException("device handle is null", -4);
    }

    private byte[] readArray(Command command, int value, int index, int length) throws LibUsbException, DeviceException {
        if (this.mDeviceHandle != null) {
            ByteBuffer buffer = ByteBuffer.allocateDirect(length);
            int transferred = LibUsb.controlTransfer(this.mDeviceHandle, (byte)-64, command.getValue(), (short)value, (short)index, buffer, 2000L);
            if (transferred < 0) {
                throw new LibUsbException("read error", transferred);
            }
            byte[] results = new byte[transferred];
            buffer.get(results);
            return results;
        }
        throw new LibUsbException("device handle is null", -4);
    }

    public void write(Command command, int value, int index, ByteBuffer buffer) throws DeviceException {
        if (this.mDeviceHandle != null) {
            int transferred = LibUsb.controlTransfer(this.mDeviceHandle, (byte)64, command.getValue(), (short)value, (short)index, buffer, 2000L);
            if (transferred < 0) {
                throw new LibUsbException("error writing byte buffer", transferred);
            }
            if (transferred != buffer.capacity()) {
                throw new LibUsbException("transferred bytes [" + transferred + "] is not what was expected [" + buffer.capacity() + "]", transferred);
            }
        } else {
            throw new LibUsbException("device handle is null", -4);
        }
    }

    @Override
    public void cleanup() throws IOException, DeviceException {
        this.stop();
    }

    @Override
    public boolean isConnected() {
        return false;
    }

    @Override
    public DevicePanel getDevicePanel() throws IOException, DeviceException {
        return new AirspyPanel();
    }

    @Override
    public void setSampleRate(RTL2832TunerController.SampleRate sampleRate) throws DeviceException {
    }

    public static enum BoardID {
        AIRSPY(0, "Airspy"),
        UNKNOWN(-1, "Unknown");

        private int mValue;
        private String mLabel;

        private BoardID(int value, String label) {
            this.mValue = value;
            this.mLabel = label;
        }

        public int getValue() {
            return this.mValue;
        }

        public String getLabel() {
            return this.mLabel;
        }

        public static BoardID fromValue(int value) {
            if (value == 0) {
                return AIRSPY;
            }
            return UNKNOWN;
        }
    }

    public class BufferDispatcher
    implements Runnable {
        @Override
        public void run() {
            try {
                ArrayList buffers = new ArrayList();
                AirspyDevice.this.mFilledBuffers.drainTo(buffers);
                for (byte[] buffer : buffers) {
                    float[] realSamples = AirspyDevice.this.mSampleAdapter.convert(buffer);
                    realSamples = AirspyDevice.this.mDCFilter.filter(realSamples);
                    float[] quadratureSamples = AirspyDevice.this.mHilbertTransform.filter(realSamples);
                    if (AirspyDevice.this.usbSource == null) continue;
                    AirspyDevice.this.usbSource.receive(quadratureSamples);
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public class BufferProcessor
    implements Runnable,
    TransferCallback {
        private ScheduledFuture<?> mSampleDispatcherTask;
        private LinkedTransferQueue<Transfer> mAvailableTransfers;
        private LinkedTransferQueue<Transfer> mTransfersInProgress = new LinkedTransferQueue();
        private AtomicBoolean mRunning = new AtomicBoolean();
        private ByteBuffer mLibUsbHandlerStatus;
        private boolean mCancel = false;

        @Override
        public void run() {
            if (this.mRunning.compareAndSet(false, true)) {
                this.prepareTransfers();
                this.mSampleDispatcherTask = AirspyDevice.this.mThreadPoolManager.scheduleFixedRate(ThreadPoolManager.ThreadType.SOURCE_SAMPLE_PROCESSING, new BufferDispatcher(), 20L, TimeUnit.MILLISECONDS);
                this.mLibUsbHandlerStatus = ByteBuffer.allocateDirect(4);
                ArrayList transfers = new ArrayList();
                this.mCancel = false;
                while (this.mRunning.get()) {
                    this.mAvailableTransfers.drainTo(transfers);
                    for (Transfer transfer : transfers) {
                        int result = LibUsb.submitTransfer(transfer);
                        if (result == 0) {
                            this.mTransfersInProgress.add(transfer);
                            continue;
                        }
                        Log.errorDialog("ERROR", "error submitting transfer [" + LibUsb.errorName(result) + "]");
                    }
                    int result = LibUsb.handleEventsTimeoutCompleted(null, 2000L, this.mLibUsbHandlerStatus.asIntBuffer());
                    if (result != 0) {
                        Log.errorDialog("ERROR", "error handling events for libusb");
                    }
                    transfers.clear();
                    this.mLibUsbHandlerStatus.rewind();
                }
                if (this.mCancel) {
                    for (Transfer transfer : this.mTransfersInProgress) {
                        LibUsb.cancelTransfer(transfer);
                    }
                    int result = LibUsb.handleEventsTimeoutCompleted(null, 2000L, this.mLibUsbHandlerStatus.asIntBuffer());
                    if (result != 0) {
                        Log.errorDialog("ERROR", "error handling events for libusb during cancel");
                    }
                    this.mLibUsbHandlerStatus.rewind();
                }
            }
        }

        public void stop() {
            this.mCancel = true;
            if (this.mRunning.compareAndSet(true, false) && this.mSampleDispatcherTask != null) {
                this.mSampleDispatcherTask.cancel(true);
                AirspyDevice.this.mFilledBuffers.clear();
            }
        }

        public boolean isRunning() {
            return this.mRunning.get();
        }

        @Override
        public void processTransfer(Transfer transfer) {
            this.mTransfersInProgress.remove(transfer);
            switch (transfer.status()) {
                case 0: 
                case 4: {
                    if (transfer.actualLength() <= 0) break;
                    ByteBuffer buffer = transfer.buffer();
                    byte[] data = new byte[transfer.actualLength()];
                    buffer.get(data);
                    buffer.rewind();
                    if (!this.isRunning()) break;
                    AirspyDevice.this.mFilledBuffers.add(data);
                    break;
                }
                case 3: {
                    break;
                }
                default: {
                    Log.errorDialog("ERROR", "transfer error [" + AirspyDevice.getTransferStatus(transfer.status()) + "] transferred actual: " + transfer.actualLength());
                }
            }
            this.mAvailableTransfers.add(transfer);
        }

        private void prepareTransfers() throws LibUsbException {
            if (this.mAvailableTransfers == null) {
                this.mAvailableTransfers = new LinkedTransferQueue();
                int x = 0;
                while (x < 16) {
                    Transfer transfer = LibUsb.allocTransfer();
                    if (transfer == null) {
                        throw new LibUsbException("couldn't allocate transfer", -11);
                    }
                    ByteBuffer buffer = ByteBuffer.allocateDirect(AirspyDevice.this.mBufferSize);
                    LibUsb.fillBulkTransfer(transfer, AirspyDevice.this.mDeviceHandle, (byte)-127, buffer, this, "Buffer", 2000L);
                    this.mAvailableTransfers.add(transfer);
                    ++x;
                }
            }
        }
    }

    public static enum Command {
        INVALID(0),
        RECEIVER_MODE(1),
        SI5351C_WRITE(2),
        SI5351C_READ(3),
        R820T_WRITE(4),
        R820T_READ(5),
        SPIFLASH_ERASE(6),
        SPIFLASH_WRITE(7),
        SPIFLASH_READ(8),
        BOARD_ID_READ(9),
        VERSION_STRING_READ(10),
        BOARD_PART_ID_SERIAL_NUMBER_READ(11),
        SET_SAMPLE_RATE(12),
        SET_FREQUENCY(13),
        SET_LNA_GAIN(14),
        SET_MIXER_GAIN(15),
        SET_VGA_GAIN(16),
        SET_LNA_AGC(17),
        SET_MIXER_AGC(18),
        MS_VENDOR_COMMAND(19),
        SET_RF_BIAS_COMMAND(20),
        GPIO_WRITE(21),
        GPIO_READ(22),
        GPIO_DIR__WRITE(23),
        GPIO_DIR_READ(24),
        GET_SAMPLE_RATES(25),
        SET_PACKING(26);

        private int mValue;

        private Command(int value) {
            this.mValue = value;
        }

        public byte getValue() {
            return (byte)this.mValue;
        }

        public static Command fromValue(int value) {
            if (value >= 0 && value <= 25) {
                return Command.values()[value];
            }
            return INVALID;
        }
    }

    public static enum GPIOPin {
        PIN_0(0),
        PIN_1(1),
        PIN_2(2),
        PIN_3(3),
        PIN_4(4),
        PIN_5(5),
        PIN_6(6),
        PIN_7(7),
        PIN_8(8),
        PIN_9(9),
        PIN_10(10),
        PIN_11(11),
        PIN_12(12),
        PIN_13(13),
        PIN_14(14),
        PIN_15(15),
        PIN_16(16),
        PIN_17(17),
        PIN_18(18),
        PIN_19(19),
        PIN_20(20),
        PIN_21(21),
        PIN_22(22),
        PIN_23(23),
        PIN_24(24),
        PIN_25(25),
        PIN_26(26),
        PIN_27(27),
        PIN_28(28),
        PIN_29(29),
        PIN_30(30),
        PIN_31(31);

        private int mValue;

        private GPIOPin(int value) {
            this.mValue = value;
        }

        public int getValue() {
            return this.mValue;
        }
    }

    public static enum GPIOPort {
        PORT_0(0),
        PORT_1(1),
        PORT_2(2),
        PORT_3(3),
        PORT_4(4),
        PORT_5(5),
        PORT_6(6),
        PORT_7(7);

        private int mValue;

        private GPIOPort(int value) {
            this.mValue = value;
        }

        public int getValue() {
            return this.mValue;
        }
    }

    public static final class Gain
    extends Enum<Gain> {
        public static final /* enum */ Gain LINEARITY_1 = new Gain(1, 4, 0, 0);
        public static final /* enum */ Gain LINEARITY_2 = new Gain(2, 5, 0, 0);
        public static final /* enum */ Gain LINEARITY_3 = new Gain(3, 6, 1, 0);
        public static final /* enum */ Gain LINEARITY_4 = new Gain(4, 7, 1, 0);
        public static final /* enum */ Gain LINEARITY_5 = new Gain(5, 8, 1, 0);
        public static final /* enum */ Gain LINEARITY_6 = new Gain(6, 9, 1, 0);
        public static final /* enum */ Gain LINEARITY_7 = new Gain(7, 10, 2, 0);
        public static final /* enum */ Gain LINEARITY_8 = new Gain(8, 10, 2, 1);
        public static final /* enum */ Gain LINEARITY_9 = new Gain(9, 10, 0, 3);
        public static final /* enum */ Gain LINEARITY_10 = new Gain(10, 10, 0, 5);
        public static final /* enum */ Gain LINEARITY_11 = new Gain(11, 10, 1, 6);
        public static final /* enum */ Gain LINEARITY_12 = new Gain(12, 10, 0, 8);
        public static final /* enum */ Gain LINEARITY_13 = new Gain(13, 10, 0, 9);
        public static final /* enum */ Gain LINEARITY_14 = new Gain(14, 10, 5, 8);
        public static final /* enum */ Gain LINEARITY_15 = new Gain(15, 10, 6, 9);
        public static final /* enum */ Gain LINEARITY_16 = new Gain(16, 11, 6, 9);
        public static final /* enum */ Gain LINEARITY_17 = new Gain(17, 11, 7, 10);
        public static final /* enum */ Gain LINEARITY_18 = new Gain(18, 11, 8, 12);
        public static final /* enum */ Gain LINEARITY_19 = new Gain(19, 11, 9, 13);
        public static final /* enum */ Gain LINEARITY_20 = new Gain(20, 11, 11, 14);
        public static final /* enum */ Gain LINEARITY_21 = new Gain(21, 12, 12, 14);
        public static final /* enum */ Gain LINEARITY_22 = new Gain(22, 13, 12, 14);
        public static final /* enum */ Gain SENSITIVITY_1 = new Gain(1, 4, 0, 0);
        public static final /* enum */ Gain SENSITIVITY_2 = new Gain(2, 4, 0, 1);
        public static final /* enum */ Gain SENSITIVITY_3 = new Gain(3, 4, 0, 2);
        public static final /* enum */ Gain SENSITIVITY_4 = new Gain(4, 4, 0, 3);
        public static final /* enum */ Gain SENSITIVITY_5 = new Gain(5, 4, 1, 5);
        public static final /* enum */ Gain SENSITIVITY_6 = new Gain(6, 4, 2, 6);
        public static final /* enum */ Gain SENSITIVITY_7 = new Gain(7, 4, 2, 7);
        public static final /* enum */ Gain SENSITIVITY_8 = new Gain(8, 4, 3, 8);
        public static final /* enum */ Gain SENSITIVITY_9 = new Gain(9, 4, 4, 9);
        public static final /* enum */ Gain SENSITIVITY_10 = new Gain(10, 5, 4, 9);
        public static final /* enum */ Gain SENSITIVITY_11 = new Gain(11, 5, 4, 12);
        public static final /* enum */ Gain SENSITIVITY_12 = new Gain(12, 5, 7, 12);
        public static final /* enum */ Gain SENSITIVITY_13 = new Gain(13, 5, 8, 13);
        public static final /* enum */ Gain SENSITIVITY_14 = new Gain(14, 5, 9, 14);
        public static final /* enum */ Gain SENSITIVITY_15 = new Gain(15, 6, 9, 14);
        public static final /* enum */ Gain SENSITIVITY_16 = new Gain(16, 7, 10, 14);
        public static final /* enum */ Gain SENSITIVITY_17 = new Gain(17, 8, 10, 14);
        public static final /* enum */ Gain SENSITIVITY_18 = new Gain(18, 9, 11, 14);
        public static final /* enum */ Gain SENSITIVITY_19 = new Gain(19, 10, 12, 14);
        public static final /* enum */ Gain SENSITIVITY_20 = new Gain(20, 11, 12, 14);
        public static final /* enum */ Gain SENSITIVITY_21 = new Gain(21, 12, 12, 14);
        public static final /* enum */ Gain SENSITIVITY_22 = new Gain(22, 13, 12, 14);
        public static final /* enum */ Gain CUSTOM = new Gain(1, 0, 0, 0);
        private int mValue;
        private int mIF;
        private int mMixer;
        private int mLNA;
        private static final /* synthetic */ Gain[] ENUM$VALUES;

        static {
            ENUM$VALUES = new Gain[]{LINEARITY_1, LINEARITY_2, LINEARITY_3, LINEARITY_4, LINEARITY_5, LINEARITY_6, LINEARITY_7, LINEARITY_8, LINEARITY_9, LINEARITY_10, LINEARITY_11, LINEARITY_12, LINEARITY_13, LINEARITY_14, LINEARITY_15, LINEARITY_16, LINEARITY_17, LINEARITY_18, LINEARITY_19, LINEARITY_20, LINEARITY_21, LINEARITY_22, SENSITIVITY_1, SENSITIVITY_2, SENSITIVITY_3, SENSITIVITY_4, SENSITIVITY_5, SENSITIVITY_6, SENSITIVITY_7, SENSITIVITY_8, SENSITIVITY_9, SENSITIVITY_10, SENSITIVITY_11, SENSITIVITY_12, SENSITIVITY_13, SENSITIVITY_14, SENSITIVITY_15, SENSITIVITY_16, SENSITIVITY_17, SENSITIVITY_18, SENSITIVITY_19, SENSITIVITY_20, SENSITIVITY_21, SENSITIVITY_22, CUSTOM};
        }

        private Gain(int value, int ifGain, int mixer, int lna) {
            this.mValue = value;
            this.mIF = ifGain;
            this.mMixer = mixer;
            this.mLNA = lna;
        }

        public int getValue() {
            return this.mValue;
        }

        public int getIF() {
            return this.mIF;
        }

        public int getMixer() {
            return this.mMixer;
        }

        public int getLNA() {
            return this.mLNA;
        }

        public static Gain getGain(GainMode mode, int value) {
            assert (1 <= value && value <= 22);
            switch (mode) {
                case LINEARITY: {
                    for (Gain gain : Gain.getLinearityGains()) {
                        if (gain.getValue() != value) continue;
                        return gain;
                    }
                    return LINEARITY_GAIN_DEFAULT;
                }
                case SENSITIVITY: {
                    for (Gain gain : Gain.getSensitivityGains()) {
                        if (gain.getValue() != value) continue;
                        return gain;
                    }
                    return SENSITIVITY_GAIN_DEFAULT;
                }
            }
            return CUSTOM;
        }

        public static GainMode getGainMode(Gain gain) {
            if (gain == CUSTOM) {
                return GainMode.CUSTOM;
            }
            if (Gain.getLinearityGains().contains((Object)gain)) {
                return GainMode.LINEARITY;
            }
            if (Gain.getSensitivityGains().contains((Object)gain)) {
                return GainMode.SENSITIVITY;
            }
            return GainMode.CUSTOM;
        }

        public static EnumSet<Gain> getLinearityGains() {
            return EnumSet.range(LINEARITY_1, LINEARITY_22);
        }

        public static EnumSet<Gain> getSensitivityGains() {
            return EnumSet.range(SENSITIVITY_1, SENSITIVITY_22);
        }

        public static Gain[] values() {
            Gain[] gainArray = ENUM$VALUES;
            int n = gainArray.length;
            Gain[] gainArray2 = new Gain[n];
            System.arraycopy(ENUM$VALUES, 0, gainArray2, 0, n);
            return gainArray2;
        }

        public static Gain valueOf(String string) {
            return Enum.valueOf(Gain.class, string);
        }
    }

    public static enum GainMode {
        LINEARITY,
        SENSITIVITY,
        CUSTOM;

    }

    public static enum ReceiverMode {
        OFF(0),
        ON(1);

        private int mValue;

        private ReceiverMode(int value) {
            this.mValue = value;
        }

        public int getValue() {
            return this.mValue;
        }
    }
}

